using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Data;
using System.Text.RegularExpressions;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.GUI.controls
{
	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/11/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///Class AboRHComboBox
	///User control which extends combo box to allow users to enter an ABO/Rh value by selecting from list,
	///or by scanning a Codabar or ISBT-128 barcode
	///</summary>
	public class AboRHComboBox : gov.va.med.vbecs.GUI.controls.ComboBoxLookup//, ISite
	{
		private Common.ABO _ABO = Common.ABO.NA;
		private Common.RH _RH = Common.RH.NotProvided;

		/// <summary>
		/// isbtScan object
		/// </summary>
		public Common.ISBTScan isbtScan = Common.ISBTScan.Unknown;

		private int _donationTypeId = -1;
		private bool barcode = false;
		private bool _Valid = true;

		private Regex _codabarABORh;

		/// <summary>
		/// Raises event if a barcode is scanned into this text box that is not an ABO/Rh code.
		/// </summary>
		public event EventHandler OtherIsbtFieldScanned;
		/// <summary>
		/// Determines if product code is ISBT-128 or Codabar.
		/// </summary>
		private Common.BarcodeType barcodeType = Common.BarcodeType.None;
		/// <summary>
		/// Stores the ISBT string in cases where the wrong field is scanned.
		/// </summary>
		public string otherIsbtString;
		/// <summary>
		/// EventHandler for Barcode changed event
		/// </summary>
		public event EventHandler BarcodeChanged;	
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		/// <summary>
		/// Barcode type changed event
		/// </summary>
		public event EventHandler BarcodeTypeChanged;

		/// <summary>
		/// ABO/Rh found event
		/// </summary>
		public event EventHandler AboRHFound;

		/// <summary>
		/// ABO/Rh not found event
		/// </summary>
		public event EventHandler AboRHNotFound;
		private DataTable _dt;


		/// <summary>
		/// Raises Barcode indicator changed event
		/// </summary>
		protected virtual void OnBarcodeChanged()
		{
			if(BarcodeChanged != null)
				BarcodeChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// Raises barcode type changed evennt
		/// </summary>
		protected virtual void OnBarcodeTypeChanged()
		{
			if(BarcodeTypeChanged != null)
				BarcodeTypeChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// Raises ABO/Rh bot found event
		/// </summary>
		protected virtual void OnAboRHNotFound()
		{
			if(AboRHNotFound != null)
				AboRHNotFound(this, EventArgs.Empty);
		}

		/// <summary>
		/// Raises ABO/Rh found event
		/// </summary>
		protected virtual void OnAboRHFound()
		{
			if(AboRHFound != null)
				AboRHFound(this, EventArgs.Empty);
		}

		/// <summary>
		/// Raises event if a barcode is scanned into this text box that is not an ABO\Rh value.
		/// </summary>
		protected virtual void OnOtherIsbtFieldScanned()
		{
			if(OtherIsbtFieldScanned != null)
				OtherIsbtFieldScanned(this, EventArgs.Empty);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		/// <summary>
		/// AboRHComboBox constructor. Creates an instance of the control and 
		/// loads all possible ABO/Rh values. Adds the control to the existing
		/// container.
		/// </summary>
		public AboRHComboBox(System.ComponentModel.IContainer container)
		{

			container.Add(this);
			InitializeComponent();
			LoadValues();

			this._codabarABORh = Common.RegularExpressions.CodabarABORh();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		/// <summary>
		/// AboRHComboBox constructor. Creates an instance of the control and 
		/// loads all possible ABO/Rh values.
		/// </summary>
		public AboRHComboBox()
		{
			InitializeComponent();
			this._codabarABORh = Common.RegularExpressions.CodabarABORh();
		}

		/// <summary> Description: Clean up any resources being used.</summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			
		}
		#endregion

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		/// <summary>
		/// Gets blood type of selected entry
		/// </summary>
		public Common.ABO Abo
		{
			get
			{
				return this._ABO;
			}
		}

		/// <summary>
		/// Get the Valid indicator
		/// </summary>
		public bool AboRHValid
		{
			get
			{
				return this._Valid;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		/// <summary>
		/// Gets Rh factor of selected entry
		/// </summary>
		public Common.RH RH
		{
			get
			{
				return this._RH;
			}
		}


		/// <summary>
		/// DonationType's primary key
		/// </summary>
		public int DonationTypeId
		{
			get
			{
				return this._donationTypeId;
			}
			set
			{
				this._donationTypeId = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		/// <summary>
		/// Gets or sets Barcode property. True if entry was made by scanning a barcode.
		/// Setting raises change event.
		/// </summary>
		[
		Category("Behavior"),
		Description("Barcode"),
		Bindable(true),
		Browsable(true),
		]
		public bool Barcode
		{
			get
			{
				return this.barcode;
			}
			set
			{        
				this.barcode = value;
				OnBarcodeChanged();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		/// <summary>
		/// Gets or sets barcode type property. BarcodeType enum of blood product code entered.
		/// </summary>
		[
		Category("Behavior"),
		Description("Barcode Type"),
		Bindable(true),
		Browsable(true),
		]
		public Common.BarcodeType BarcodeType
		{
			get
			{
				return barcodeType; 
			}
			set
			{
				barcodeType = value;
				if (this.barcode)
					//Scanned
				{
					this.GetABORhFromCode();				
				}

				OnBarcodeTypeChanged();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		/// <summary>
		/// Override for KeyPress event. Checks for ~ character which denotes a barcode was scanned.
		/// </summary>
		/// <param name="e"></param>
		protected override void OnKeyPress(System.Windows.Forms.KeyPressEventArgs e)
		{
			base.OnKeyPress(e);
			if (e.KeyChar == '~')
			{
				this.Barcode = true;
				this.Text = this.Text.Replace("~",string.Empty);
				e.Handled = true;
				return;
			}
			else if (e.KeyChar == (char)13)
			{
				this.Barcode = false;
				e.Handled = true;
				return;
			}
		}
		private static bool Contains(string[] columnList,string comparisonId)
		{
			foreach(string col in columnList) 
			{
				if(col == comparisonId) 
				{
					return true;
				}
			}
			return false;
		}
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		/// <summary>
		/// Override for OnLeave event. Determines barcode type, if applicable
		/// </summary>
		/// <param name="e"></param>
		protected override void OnLeave(EventArgs e)
		{
			string sText = this.Text;
			isbtScan = Common.Utility.GetISBTFromString(this.Text);
			this.otherIsbtString = this.Text;
			this._Valid = true;

			
			const string ABORhText = "O Pos,A Pos,B Pos,AB Pos,O Neg,A Neg,B Neg,AB Neg,O,A,B,AB,Pooled Pos, Pooled Neg,O Pooled,A Pooled, B Pooled, AB Pooled, Pooled Pooled";
			if(Contains(ABORhText.Split(','), sText))
			{
				this.Barcode = false;
			}

			//Scanned
			if (this.Barcode)
			{
				if(this.Text.Length >= 2)
				{
					//BR_1.21
					if (this.Text.StartsWith("=%"))
					{
						this.Text = this.Text.Replace("=%",string.Empty);
						this.BarcodeType =Common.BarcodeType.ISBT128;
					}
					else
					{
						if(sText.Length >=3)
						{
							string sFirstChar = sText.Substring(0, 1);
							string sSecondChar = sText.Substring(1,1);
							string sThirdChar = sText.Substring(2,1);
							
							try
							{
								System.Convert.ToInt32(sFirstChar);
							}
							catch
							{
								_Valid = false;
							}
							try
							{
								System.Convert.ToInt32(sSecondChar);
							}
							catch
							{
								_Valid = false;
							}
							try
							{
								System.Convert.ToInt32(sThirdChar);
							}
							catch
							{
								_Valid = false;
							}
						}
						this.BarcodeType =Common.BarcodeType.Codabar;
					}
				}
				if(this.BarcodeType == Common.BarcodeType.ISBT128)
				{
					if(sText.Trim().Length != 6)
					{
						this._Valid = false;
					}
					else
					{
						this._Valid = true;
					}
				}
				else
				{
					if(sText.Trim().Length != 3)
					{
						this._Valid = false;
					}
					else
					{
						this._Valid = true;
					}
				}
				
			}
				//Typed in
			else
			{
				if (this.SelectedIndex > 0)
				{
					this.OnAboRHFound();
				}
				else
				{
					this.OnAboRHNotFound();
				}
			}

			if (isbtScan != Common.ISBTScan.ABORh && isbtScan != Common.ISBTScan.Unknown)
			{
				this._Valid = false;
				OnOtherIsbtFieldScanned();			
			}

			base.OnLeave(e);
		}

		/// <summary>
		/// Recognizes the barcode type and set the appropriate property of the control based
		/// on it. Then retrieves the ABORh from the barcode scanned code.
		/// </summary>
		/// <param name="controlText"></param>
		public void SetControl(string controlText)
		{
			this.Barcode = true;
			//this.Text = inputText;
			
			if(controlText.Length >= 2)
			{
				//BR 1.21
				if (controlText.StartsWith("=%"))
				{
					controlText = controlText.Replace("=%",string.Empty);
					this.BarcodeType =Common.BarcodeType.ISBT128;
					this.GetABORhFromCode(controlText);
				}
				else
				{
					this.BarcodeType =Common.BarcodeType.Codabar;
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Private method to determine actual ABO/Rh value from barcode scanned code
		/// </summary>
		private void GetABORhFromCode()
		{
			//Calling BR 1.20, TT_1.02
			try
			{
				if ((this.Text.Length >= 2) && (_codabarABORh.IsMatch(this.Text.ToUpper())))
				{
					string _txt = this.Text.ToUpper().Substring(0,2);
					DataRow dr = BOL.BloodUnit.GetBloodTypeRHFactorFromBloodTypeCode(_txt);

					string aboString = dr[ARTIFICIAL.BloodType].ToString().Trim();
					string rhString = dr[TABLE.RhFactor.RhFactorCode].ToString().Trim();
					string aborhString = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Trim();
					//This is code fix that has been postponed till v2.0, CMC changed mind 
							//and assigned CR 1594 to 1.6 release
					if(aborhString.IndexOf("Mx") > -1 )
					{
						aborhString = aborhString.Replace("Mx", "Pooled");
					}

					this.Text = aborhString.Trim();
				

					this._ABO = BOL.AboRh.GetAboFromString(aboString);
					this._RH = BOL.AboRh.GetRHFromCode(rhString);
					OnAboRHFound();
				}
			}
			catch (Exception)
			{
				this.Text = string.Empty;
				this._ABO = Common.ABO.NA;
				this._RH = Common.RH.NotProvided;
                OnAboRHNotFound();
            }
		}

	
		/// <summary>
		/// Private method to determine actual ABO/Rh value from barcode scanned code
		/// </summary>
		private void GetABORhFromCode(string controlText)
		{
			//Callin BR 1.20
			try
			{
				if ((controlText.Length >= 2) && (_codabarABORh.IsMatch(controlText.ToUpper())))
				{
					string _txt = controlText.ToUpper().Substring(0,2);
					DataRow dr = BOL.BloodUnit.GetBloodTypeRHFactorFromBloodTypeCode(_txt);

					string aboString = dr[ARTIFICIAL.BloodType].ToString();
					string rhString = dr[TABLE.RhFactor.RhFactorCode].ToString();
					string aborhString = dr[ARTIFICIAL.BloodTypeRhFactor].ToString();

					this.Text = aborhString;

					this._ABO = BOL.AboRh.GetAboFromString(aboString);
					this._RH = BOL.AboRh.GetRHFromCode(rhString);
					OnAboRHFound();
				}
			}
			catch (Exception)
			{
				this.Text = string.Empty;
				this._ABO = Common.ABO.NA;
				this._RH = Common.RH.NotProvided;
                OnAboRHNotFound();
            }
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///
		/// <summary>
		/// Selected index changed event handler. Sets ABO and Rh properties based on selected entry.
		/// </summary>
		/// <param name="e">Event args</param>
		protected override void OnSelectedIndexChanged(EventArgs e)
		{
			if (this.SelectedIndex>0)
			{
				//
				//string _text = this.Text;
				string _text = this.Text.ToString();
				//Delete
				string[] aborh = _text.Split(' ');
				//
				if (aborh.Length > 0) 
				{
					this._ABO = BOL.AboRh.GetAboFromString(aborh[0]);
				}
				if (aborh.Length > 1)
				{
					foreach(DataRow dr in _dt.Rows)
					{
						string sItem  = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
						if(sItem.Trim() == _text.Trim())
						{
							this._RH = BOL.AboRh.GetRHFromCode(dr[TABLE.RhFactor.RhFactorCode].ToString());
							return;
						}
					}
				}
				else
				{
					this._RH = Common.RH.Blank;
				}
				//
			}
			else
			{
				this._ABO = Common.ABO.NA;
				this._RH = Common.RH.NotProvided;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		/// <summary>
		/// Public method to load combo box with all possible ABO/Rh values.
		/// If put in the constructor, causes long load times.
		/// Should be called after initialization.
		/// </summary>
		public void LoadValues()
		{
			this.Items.Clear();
			this.Items.Add(string.Empty);

			_dt = BOL.AboRh.GetAllAboRH();

			foreach(DataRow dr in _dt.Rows)
			{
				string sItem  = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
				if(sItem != string.Empty)
				{
					this.Items.Add(sItem);
				}
			}
		}
		private void SortDataTable(ref System.Data.DataTable dtNew, string sortItem)
		{
			foreach(System.Data.DataRow drNewRow in _dt.Rows)
			{
				string sItem  = drNewRow[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
				if(sItem.Trim() == sortItem.Trim())
				{
					System.Data.DataRow newRow = dtNew.NewRow();
					newRow.ItemArray = drNewRow.ItemArray;
					dtNew.Rows.Add(newRow);
					break;
				}
			}		
		}

		private static bool Include(string[] columnList,string comparisonId)
		{
			foreach(string col in columnList) 
			{
				if(col == comparisonId) 
				{
					return true;
				}
			}
			return false;
		}
		/// <summary>
		/// Implements BR_1.66, TT_1.02, TT_1.09 to load values based on product type
		/// </summary>
		/// <param name="bloodProduct" type="gov.va.med.vbecs.BOL.BloodProduct">
		///     <para>
		///         
		///     </para>
		/// </param>
		/// <returns>
		///     A void value...
		/// </returns>
		public void LoadValues(BOL.BloodProduct bloodProduct)
		{
			this.Items.Clear();
			this.Items.Add(string.Empty);

			_dt = BOL.AboRh.GetAllAboRH();

			System.Data.DataTable dtNew = _dt.Clone();

			SortDataTable(ref dtNew, "O Pos");
			SortDataTable(ref dtNew, "A Pos");
			SortDataTable(ref dtNew, "B Pos");
			SortDataTable(ref dtNew, "AB Pos");
			SortDataTable(ref dtNew, "O Neg");
			SortDataTable(ref dtNew, "A Neg");
			SortDataTable(ref dtNew, "B Neg");
			SortDataTable(ref dtNew, "AB Neg");
			SortDataTable(ref dtNew, "O");
			SortDataTable(ref dtNew, "A");
			SortDataTable(ref dtNew, "B");
			SortDataTable(ref dtNew, "AB");
			SortDataTable(ref dtNew, "Pooled Pos");
			SortDataTable(ref dtNew, "Pooled Neg");
			SortDataTable(ref dtNew, "O Pooled");
			SortDataTable(ref dtNew, "A Pooled");
			SortDataTable(ref dtNew, "B Pooled");
			SortDataTable(ref dtNew, "AB Pooled");
			SortDataTable(ref dtNew, "Pooled Pooled");

			//TT_1.09
			string[] includeProductTypeCodes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WholeBlood),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRejuvenatedRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RejuvenatedRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RejuvenatedApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRejuvenatedApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PlateletRichPlasma),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Platelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedPlatelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisPlatelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenApheresisPlatelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisPlatelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisPlatelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Granulocytes),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisGranulocytes),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisGranulocytesPlatelets),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Leukocytes),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisLeukocytes),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PlateletRichBuffyCoat),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisLymphocytes),
												   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisMonocytes)};
			//
			if(Include(includeProductTypeCodes, bloodProduct.ProductType.ProductTypeCode))
			{
				foreach(DataRow dr in dtNew.Rows)
				{
					string sItem  = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
					sItem = sItem.Trim();
					if(sItem != string.Empty)
					{
						if(sItem == "O Pos" || sItem == "A Pos" || 
							sItem == "B Pos" || sItem == "AB Pos" || 
							sItem == "O Neg" || sItem == "A Neg" || 
							sItem == "B Neg" || sItem == "AB Neg" )
						{
							this.Items.Add(sItem);
						}
					}
				}
				return;
			}
			//
			includeProductTypeCodes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FreshFrozenPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedFreshFrozenPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisFreshFrozenPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisFreshFrozenPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Cryoprecipitate),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedCryoprecipitate),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisCryoprecipitate),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisCryoprecipitate),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Serum),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.LiquidPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Plasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisPlasma),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.LiquidApheresisPlasma)};
			//
			if(Include(includeProductTypeCodes, bloodProduct.ProductType.ProductTypeCode))
			{
				foreach(DataRow dr in dtNew.Rows)
				{
					string sItem  = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
					sItem = sItem.Trim();
					if(sItem != string.Empty)
					{
						if(sItem == "O Pos" || sItem == "A Pos" || 
							sItem == "B Pos" || sItem == "AB Pos" || 
							sItem == "O Neg" || sItem == "A Neg" || 
							sItem == "B Neg" || sItem == "AB Neg" ||
							sItem == "O" || sItem == "A" ||
							sItem == "B" || sItem == "AB")
						{
							this.Items.Add(sItem);
						}
					}
				}
				return;
			}

			includeProductTypeCodes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledPlatelets),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedPooledPlatelets),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledGranulocytes),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledPlateletRichBuffyCoat)};
			//
			if(Include(includeProductTypeCodes, bloodProduct.ProductType.ProductTypeCode))
			{
				foreach(DataRow dr in dtNew.Rows)
				{
					string sItem  = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
					sItem = sItem.Trim();
					if(sItem != string.Empty)
					{
						if(sItem == "O Pos" || sItem == "A Pos" || 
							sItem == "B Pos" || sItem == "AB Pos" || 
							sItem == "O Neg" || sItem == "A Neg" || 
							sItem == "B Neg" || sItem == "AB Neg" ||
							sItem == "Pooled Pos" || sItem == "Pooled Neg" ||
							sItem == "O Pooled" || sItem == "A Pooled" ||
							sItem == "B Pooled" || sItem == "AB Pooled" ||
							sItem == "Pooled Pooled")
						{
							this.Items.Add(sItem);
						}
					}
				}
				return;
			}

			includeProductTypeCodes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledCryoprecipitate),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedPooledCryoprecipitate),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledSerum),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenPooledSerum),
													  Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledPlasma)};
			//
			if(Include(includeProductTypeCodes, bloodProduct.ProductType.ProductTypeCode))
			{
				foreach(DataRow dr in dtNew.Rows)
				{
					string sItem  = dr[ARTIFICIAL.BloodTypeRhFactor].ToString().Replace("Mx", "Pooled");
					sItem = sItem.Trim();
					if(sItem != string.Empty)
					{
						this.Items.Add(sItem);
					}
				}
				return;
			}
		}
	}
}

